<?php
/**
 * @package			No Boss Extensions
 * @subpackage  	No Boss Calendar
 * @author			No Boss Technology <contact@nobosstechnology.com>
 * @copyright		Copyright (C) 2021 No Boss Technology. All rights reserved.
 * @license			GNU Lesser General Public License version 3 or later; see <https://www.gnu.org/licenses/lgpl-3.0.en.html>
 */

defined('_JEXEC') or die();

/*
 * Classe com requisicoes de apis externas (google calendar, etc)
 */
trait ModNobosscalendarHelperApis {
    /**
	 * Obtem access token valido para google calendar
     * 
     * @param   String      $clientId       Client ID para caso precise obter novamente o token de acesso
     * @param   String      $clientSecret   Secret para cado precise obter novamente o token de acesso
     * @param   String      $accessToken    Token de acesso para requisicao na API
     * @param   String      $refreshToken   Token de renovacao para caso precise obter novamente o token de acesso
     * @param   Int         $idModule       Id do modulo que estamos executando (optional)
	 *
	 * @return  Mixed       access_token valido ou excecao
	 */
    public static function getGoogleValidAccessToken($clientId, $clientSecret, $accessToken, $refreshToken, $idModule = false){
        jimport('noboss.api.googlecalendar');
        
        if(empty($accessToken) || empty($refreshToken)){
            return false;
        }

        // Obtem access token valido
        $accessToken = NoBossApiGooglecalendar::getValidAccessToken($clientId, $clientSecret, $refreshToken, $accessToken);

        if(!$accessToken){
            throw new \Exception('Could not generate access_token from refresh_token'); 
        }

        // Id de modulo definido: atualiza o access token no banco de dados do modulo
        if(!empty($idModule)){
            require_once JPATH_LIBRARIES.'/noboss/forms/fields/nobossapiconnection/nobossapiconnectionhelper.php';
            NobossNobossapiconnectionhelper::updateAccessTokenDb($accessToken, $idModule, 'google_calendar', true);
        }

        return $accessToken;
    }

    /**
     * Obtem agendas do google calendar vinculadas a um token
	 *
     * @return void.
	 */
    public static function apiGetSchedulesAjax(){
        error_reporting(0);

        jimport('noboss.api.googlecalendar');
        
		$app = JFactory::getApplication();
        
        // Obtem dados da api
        $clientId = $app->input->get("client_id", "", "RAW");
        $clientSecret = $app->input->get("client_secret", "", "RAW");
        $accessToken = $app->input->get("access_token", "", "RAW");
        $refreshToken = $app->input->get("refresh_token", "", "RAW");

        // Obtem qual eh a api que esta sendo trabalhada (ex: googlecalendar)
        // TODO: usar para diferenciar no futuro o google calendar do outlook
        $api = $app->input->get("api", "", "RAW");
    
        try{
            // Obtem access token valido
            $accessToken = self::getGoogleValidAccessToken($clientId, $clientSecret, $accessToken, $refreshToken);
        }
        catch(Exception $e){
            $response['success'] = 0;
            $response['message'] = '';
            if(!empty($e->getCode())){
                $response['message'] = "Code: ".$e->getCode();
            }
            $response['message'] .= $e->getMessage();
            exit(json_encode($response));
        }

        try{
            // Obtem agendas do calendar
            $schedules = NoBossApiGooglecalendar::getCalendars($accessToken);
        }
        catch(Exception $e){
            $response['success'] = 0;
            $response['message'] = '';
            if(!empty($e->getCode())){
                $response['message'] = "Code: ".$e->getCode();
            }
            $response['message'] .= $e->getMessage();
            exit(json_encode($response));
        }

        $response['success'] = 1;
        $response['items'] = $schedules;
        exit(json_encode($response));
    }

    /**
     * Realiza importacao manual de eventos de API externa (executado a partir do admin do modulo)
	 *
     * @return void.
	 */
    public static function apiImportManualAjax(){
        error_reporting(0);

        $app = JFactory::getApplication();

        // Obtem o id do modulo (se for edicao) e id do calendar
        $idModule = $app->input->get("id_module", "", "INT");
        $idCalendar = $app->input->get("id_calendar", "", "STRING");
        $idCalendarOriginal = $app->input->get("id_calendar_original", "", "STRING");

        // Obtem parametros da api
        $paramsApi = urldecode($app->input->get("params_api", "", "RAW"));
        $paramsApi = json_decode($paramsApi);

        // Agendas selecionadas
        $schedules = json_decode($app->input->get("schedules", "", "RAW"));
        
        // Regras dos campos que devem ser importados
        $rules = json_decode($app->input->get("rules", "", "RAW"));
        
        try{
            // Periodos para importacao
            $startDate = $app->input->get("start_date", "", "RAW");
            $finalDate = $app->input->get("final_date", "", "RAW");
            // Passa pelo Joomla para ajustar data qnd informado dia que nao existe no mes e para identificar data invalida
            $startDate = JFactory::getDate($startDate, 'UTC')->format('Y-m-d'); 
            $finalDate = JFactory::getDate($finalDate, 'UTC')->format('Y-m-d'); 
        }
        catch(Exception $e){
            // Formato de data incorreto
            $response['success'] = 0;
            $response['message'] = 'Dates are invalid';
            exit(json_encode($response));
        }

        // Obtem qual eh a api que esta sendo trabalhada (ex: googlecalendar)
        // TODO: usar para diferenciar no futuro o google calendar do outlook
        $api = $app->input->get("api", "", "RAW");

        //echo "Data inicial: ".$startDate->format('d/m/Y - H:i')." | Data final: ".$finalDate->format('d/m/Y - H:i'); exit;

        try{
            // Obtem access token valido
            $accessToken = self::getGoogleValidAccessToken($paramsApi->client_id, $paramsApi->client_secret, $paramsApi->access_token, $paramsApi->refresh_token, $idModule);
        }
        catch(Exception $e){
            $response['success'] = 0;
            $response['message'] = '';
            if(!empty($e->getCode())){
                $response['message'] = "Code: ".$e->getCode();
            }
            $response['message'] .= $e->getMessage();
            exit(json_encode($response));
        }

        try{
            // Executa funcao que busca eventos na api e atualiza na base local
            $eventsImport = self::googleImportEvents($startDate, $finalDate, $accessToken, $schedules, (array) $rules, $idCalendar, $idModule, $idCalendarOriginal);
        }
        catch(Exception $e){
            $response['success'] = 0;
            $response['message'] = '';
            if(!empty($e->getCode())){
                $response['message'] = "Code: ".$e->getCode();
            }
            $response['message'] .= $e->getMessage();
            exit(json_encode($response));
        } 

        // Retorna sucesso e a data que importacao foi realizada
        $response['success'] = 1;
        $response['date'] = JFactory::getDate('now', 'UTC')->format('Y-m-d H:i');
        $response['events_import'] = $eventsImport;
        exit(json_encode($response));
    }

    /**
     * Realiza importacao automatica de eventos de api externa (executado a partir de js no front do modulo)
	 *
     * @return void.
	 */
    public static function apiImportAutomaticAjax(){
        error_reporting(0);

        $app = JFactory::getApplication();
        // Obtem configuracoes globais
        $config     = JFactory::getConfig();

        // Obtem o id do modulo
        $idModule = $app->input->get("id_module", "", "INT");
        
        // Obtem o id do calendario
        $IdCalendar = $app->input->get("id_calendar", "", "STRING");
        
        // Obtem qual eh a api que esta sendo trabalhada (ex: googlecalendar)
        // TODO: usar para diferenciar no futuro o google calendar do outlook
        $api = $app->input->get("api", "", "RAW");

        // Cache esta habilitado: obtem params da API manualmente do banco
        if($config->get('caching', '0')){
            jimport('noboss.util.modules');
            // Obtem parametros do modulo
            $dataParams = NoBossUtilModules::getDataModule($idModule, true);
            // Extrai somente os parametros da modal de api
            $paramsApi = json_decode($dataParams->google_calendar);
        }
        // Obtem params da API via GET / POST
        else{
            // Obtem parametros da api
            $paramsApi = urldecode($app->input->get("params_api", "", "RAW"));
            $paramsApi = json_decode($paramsApi);
        }

        // Extrai parametros de conexao
        $connectionApi = json_decode($paramsApi->connection);

        // Dados necessarios nao definidos
        if (empty($paramsApi) || empty($connectionApi->access_token) || empty($connectionApi->refresh_token) || empty($paramsApi->schedules)){
            $response['success'] = 0;
            $response['message'] = 'Not all necessary data have been passed.';
            exit(json_encode($response));
        }

        $import = false;

        // Nao ha registro de nenhuma importacao automatica
        if(empty($paramsApi->importer_automatic_date)){
            // Seta para realizar importacao
            $import = true;
        }
        // Verifica se esta na hora de realizar importacao
        else{
            // Data e hora atual
            $dateNow = JFactory::getDate('now', 'UTC');
            // Data e hora da ultima importacao
            $dateLastImport = JFactory::getDate($paramsApi->importer_automatic_date, 'UTC');
            // Data e hora da proxima execucao
            $dateNextImport = clone $dateLastImport;

            // Selecionado valor 0 (zero): seta um tempo minimo de 15 minutos
            if($paramsApi->importer_automatic_interval == 0){
                $dateNextImport->modify("+15minutes");
            }
            else{
                $dateNextImport->modify("+{$paramsApi->importer_automatic_interval}hours");
            }


            // Data atual eh maior ou igual a data da proxima execucao
            if($dateNow >= $dateNextImport){
                // Seta para realizar importacao
                $import = true;
            }
            
            // Para debug para visualizar no console
            //echo "Data atual: ".$dateNow->format('d/m/Y - H:i').' | Data ultima import.: '.$dateLastImport->format('d/m/Y - H:i').' | Data proxima import.: '.$dateNextImport->format('d/m/Y - H:i'); 
        }

        // Nao deve importar eventos: apenas sai da funcao
        if(!$import){
            $response['success'] = 1;
            $response['message'] = 'Not at the time of import execution';
            exit(json_encode($response));
        }

        // Seta data inicial de importacao como primeiro dia do mes atual
        $startDate = JFactory::getDate('now', 'UTC');
        $startDate->modify('first day of this month')->modify('midnight');

        // Seta como data final de importacao o primeiro dia do mes seguinte ao mes limite que deve atualizar
        $finalDate = clone $startDate;
        $finalDate->modify("+{$paramsApi->importer_automatic_months}month")->modify('first day of this month')->modify('midnight');

        // Para debug para visualizar no console
        //echo "Data inicial importacao: ".$startDate.' | Data final importacao: '.$finalDate;

        try{
            // Obtem access token valido
            $accessToken = self::getGoogleValidAccessToken($connectionApi->client_id, $connectionApi->client_secret, $connectionApi->access_token, $connectionApi->refresh_token, $idModule);
        }
        catch(Exception $e){
            $response['success'] = 0;
            $response['message'] = '';
            if(!empty($e->getCode())){
                $response['message'] = "Code: ".$e->getCode();
            }
            $response['message'] .= $e->getMessage();
            exit(json_encode($response));
        }

        // Gera array com os campos rules, retirando a idenficacacao de rules no key ('rules_')
        $rules = array();
        foreach ($paramsApi as $key => $value) {
            if(strpos($key, 'rules_') !== false){
                $key = str_replace("rules_", "", $key);
                $rules[$key] = $value;
            }
        }

        try{
            // Executa funcao que busca eventos na api e atualiza na base local
            self::googleImportEvents($startDate->format('Y-m-d'), $finalDate->format('Y-m-d'), $accessToken, json_decode($paramsApi->schedules), $rules, $IdCalendar, $idModule);
        }
        catch(Exception $e){
            $response['success'] = 0;
            $response['message'] = '';
            if(!empty($e->getCode())){
                $response['message'] = "Code: ".$e->getCode();
            }
            $response['message'] .= $e->getMessage();
            exit(json_encode($response));
        } 

        // Atualiza nos params do modulo a data da importacao realizada
        $dateNowString = JFactory::getDate('now', 'UTC')->format('Y-m-d H:i');
        if(!self::apiUpdatesAutomaticImportDate($idModule, $dateNowString, 'google_calendar')){
            $response['success'] = 0;
            $response['message'] = 'Events were imported, but it was not possible to record the time of the import in the database.';
            exit(json_encode($response));
        }

        $response['success'] = 1;
        $response['message'] = 'Events successfully imported from Google Calendar';
        exit(json_encode($response));
    }

    /**
	 * Importa eventos de um calendario do Google para um periodo informado
     * 
     * @param   String      $startDate 		        Data inicial no formato aaaa-dd-mm
     * @param   String      $finalDate 		        Data final no formato aaaa-dd-mm
     * @param   String      $accessToken    T       oken de acesso para requisicao na API (ja validado)
     * @param   Array       $schedulesIds 	        Ids de agendas de calendario
     * @param   Array       $rules 	                Regras com campos que devem ser importados
     * @param   String      $idCalendar 	        Id do calendario (opcional, se informado idModule)
     * @param   Int         $idModule 		        Id do modulo (opcional, se informado idCalendar)
     * @param   String      $idCalendarOriginal 	Id do calendario antes do usuario supostamente alterar na pagina
	 *
	 * @return  Mixed       Array com eventos que foram importados da API ou false
	 */
    private static function googleImportEvents($startDate, $finalDate, $accessToken, $schedulesIds, $rules, $idCalendar = 0, $idModule = 0, $idCalendarOriginal = 0){
        jimport('noboss.api.googlecalendar');

        if(empty($idCalendar) && empty($idModule)){
            return false;
        }

        if(empty($idCalendarOriginal)){
            $idCalendarOriginal = $idCalendar;
        }

        // Para uso em debug do console do navegador
        //echo "Data inicial: {$startDate} | <br /> Data final: {$finalDate} | <br /> Access token: {$accessToken} | <br /> Agendas: ".json_encode($schedulesIds)."<br />";

        // Obtem agendas do calendar para poder acessar dados se necessario (para pegar cor da agenda, por exemplo)
        $schedulesData = NoBossApiGooglecalendar::getCalendars($accessToken);

        // echo '<pre>';
        // var_dump($schedulesData);
        // exit;

        // Remove todas as categorias do calendario que existam no banco para o google calendar
        self::deleteCategories($idCalendarOriginal, "googlecalendar");

        $events = array();

        // Percorre todas agendas a importar
        foreach ($schedulesIds as $scheduleId) {
            // Coloca datas no formato do Google
            $startDateGoogle = $startDate.'T00:00:00+00:00';
            $finalDateGoogle = $finalDate.'T23:59:00+00:00';

            // Obtem eventos na API para periodo e id de calendario informado
            $eventsSchedule = NoBossApiGooglecalendar::getCalendarEvents($accessToken, $scheduleId, $startDateGoogle, $finalDateGoogle);

            // Junta resultado da agenda no array de eventos
            $events = array_merge($eventsSchedule, $events);
            
            // Monta objeto com dados da categoria 
            // OBS: sempre devemos atualizar todas categorias do Google para evitar problema com eventos nao atualizados no momento
            $categorie = new stdClass();
            $categorie->id_categorie = $scheduleId;
            $categorie->category_color = $schedulesData[$scheduleId]['backgroundColor'];
            $categorie->show_in_legends = 1;
            // Quando usuario nao eh dono da agenda, ele pode ter definido uma sobreescrita do nome somente local para ele.
            $categorie->legend = (!empty($schedulesData[$scheduleId]['summaryOverride'])) ? $schedulesData[$scheduleId]['summaryOverride'] : $schedulesData[$scheduleId]['summary'];
            
            // Salva a categoria no banco de dados
            self::saveCategorie($idCalendar, $categorie, 'googlecalendar');
        }
        
        // echo 'Eventos importados: <pre>';
        // var_dump($events);
        // exit;
        
        // Remove eventos que ja estao na base de dados antes de inserir novamente
        self::apiRemovePeriodEvents('googlecalendar', $startDate, $finalDate, $idCalendarOriginal, $idModule);

        // Nao ha eventos a inserir (deixa antes remover os eventos igual pq pode ter sido apagado no google sem ter novos no mesmo periodo)
        if(empty($events)){
            return false;
        }
        
        // Obtem as cores dos eventos do google com seus codigos para fazer de-para depois
        $colorsEventsGoogle =  NoBossApiGooglecalendar::getEventsColors($accessToken);


        // Percorre os eventos retornados
        foreach ($events as $key => $eventGoogle) {
            // Objeto que armazenara os dados a serem salvos no banco
            $event = new StdClass();
            // Objeto para armazenar hora inicial e final
            $hour = new StdClass();

            // echo 'Evento: <pre>';
            // var_dump($eventGoogle);
            // exit;

            // Tipo de cadastro
            $event->input_type = 'googlecalendar';

            // Fixa como publicado
            $event->published = '1';
            
            // Titulo
            // Decodifica o html para vir correto
            $event->event_title = html_entity_decode($eventGoogle['summary']);

            // Setado para importar descricao do evento dentro da modal de conteudo
            if((!empty($rules['import_description'])) && ($rules['import_description'] != '0') && ($eventGoogle['description'])){
                // Decodifica o html para vir correto
                $eventGoogle['description'] = html_entity_decode($eventGoogle['description']);

                // Realiza quebra de linha onde tiver \n
                $eventGoogle['description'] = nl2br($eventGoogle['description']);

                // Trata a descricao limpando tags html com excecao dos passados como parametro
                $eventGoogle['description'] = strip_tags($eventGoogle['description'], '<b><a><i><br><ul><ol><li>');
                
                // Prepara para exibir como conteudo em modal
                $event->link_option = 'modal';
                $event->modal_option = 'editor';
                $event->modal_content = $eventGoogle['description'];               

                // Setado para tambem importar a descricao como texto de apoio
                if($rules['import_description'] == 'modal-and-subtitle'){
                    // Prepara conteudo do subtitulo limpando tags html com excecao dos passados como parametro
                    $event->event_subtitle = strip_tags($eventGoogle['description'], '<b><a><i><br>');

                    // Limita a quantidade de caracteres que sera colocado como subtitulo (limite definido via parametro)
                    $event->event_subtitle = JHtml::_('string.truncate', $event->event_subtitle, $rules['subtitle_limit']);
                }
            }

            // Data inicial do evento
            // Data com horario (data eh retornada no formato 2020-09-11T16:15:00-03:00)
            if(!empty($eventGoogle['start']['dateTime'])){
                $tempInitial = explode('T', $eventGoogle['start']['dateTime']);
                $event->initial_date = $tempInitial[0];
                // Ja prepara horario para salvar em seguida
                $hour->initial = substr($tempInitial[1], 0, 5);
                // Atualiza o horario com timezone recebido do google (nao estamos usando pq o joomla ta colocando 3 hrs a mais)
                //$hour->initial = JFactory::getDate("0000-00-00 {$hour->initial}");
            }
            // Nao tem definido timezone: eh data sem horario (data eh retornada no formado 2020-09-11)
            else{
                $event->initial_date = $eventGoogle['start']['date'];
            }

            // Data fina do evento
            // Data com horario (data eh retornada no formato 2020-09-11T16:15:00-03:00)
            if(!empty($eventGoogle['end']['dateTime'])){
                $tempEnd = explode('T', $eventGoogle['end']['dateTime']);
                $event->final_date = $tempEnd[0];
                // Ja prepara horario para salvar em seguida
                $hour->final = substr($tempEnd[1], 0, 5);
                // Atualiza o horario com timezone recebido do google (nao estamos usando pq o joomla ta colocando 3 hrs a mais)
                //$hour->final = JFactory::getDate("0000-00-00 {$hour->final}");
            }
            // Nao tem definido timezone: eh data sem horario (data eh retornada no formado 2020-09-11)
            else{
                /* TODO: nos eventos em que nao tem horario definido, o google tem retornado como data final sempre um dia a mais. 
                    - Isso ocorre tanto em evento de um único dia como em evento que dura mais dias.
                    - Para contornar o problema, o codigo abaixo reduz sempre um dia da data final retornada pelo google
                    - Observar se nao vamos ter problemas com clientes por causa de timezone. Pode ser que em algumas regiões o problema de retornar como um dia a mais não ocorra.
                */
                $event->final_date = JFactory::getDate($eventGoogle['end']['date'], 'UTC')->modify('-1 day')->format('Y-m-d');
            }

            // Setado para importar horario
            if((!empty($rules['import_hours'])) && (!empty($hour->initial))){
                // Salva o horario no formato que salvamos no banco de dados (array de objetos em formato json e com escapes)
                $event->hours = addslashes(json_encode(array($hour)));
            }
            
            // Define pq pode ser usado para mais de um parametro
            $event->event_place = '';

            // Setado para importar endereco
            if((!empty($rules['import_address'])) && ($eventGoogle['location'])){
                $event->event_place = $eventGoogle['location'];

                // Setado para exibir link do Google Maps
                if($rules['import_address'] == 'googlemaps'){
                    $event->event_place_link = 'googlemaps';
                }
            }

            // Setado para importar endereco do meet
            if((!empty($rules['import_meet'])) && ($eventGoogle['hangoutLink'])){
                // Ja tem algo cadastrado como localizacao: coloca separador p/ incluir link do meet
                if(!empty($event->event_place)){
                    $event->event_place .= " | ";
                }

                // Adiciona o link para o meer
                $event->event_place .= "<a href='".$eventGoogle['hangoutLink']."' class='nb-local-link' target='_blank'>Google Meet <img src='".JURI::root()."modules/mod_nobosscalendar/assets/site/images/google-meet.png'></a>";
            }

            // Setado para utilizar dados da agenda da API como categoria
            if(isset($rules['categorie_type']) && $rules['categorie_type'] == 'api'){
                // Coloca o ID da agenda do google como id de categoria
                $event->event_category = $eventGoogle['calendarID'];
            }
            // Setado para utilizar uma categoria local para todos eventos importados
            else if((isset($rules['categorie_type']) && $rules['categorie_type'] == 'local') && (!empty($rules['categorie_local']))){
                // Armazena id da categoria
                $event->event_category = $rules['categorie_local'];
            }
            // Eventos importados nao utilizam categorias
            else{
                // Codigo da cor veio no evento: pega do array qual a cor a setar
                if(!empty($eventGoogle['colorId'])){
                    $event->event_manual_color = $colorsEventsGoogle[$eventGoogle['colorId']];
                }
                // Evento nao tem cor: pegar cor da agenda
                else{
                    // Os dados das agendas estao definidos
                    if(!empty($schedulesData) && !empty($eventGoogle['calendarID'])){
                        // Obtem a cor da agenda (evento sem cor usa a cor da agenda)
                        // TODO: pegar a cor pelo 'colorid' da agenda nem sempre vinha certo. Entao mudamos para pegar direto o valor da cor em 'backgroundColor'
                        // $colorId = $schedulesData[$eventGoogle['calendarID']]['colorId'];
                        // $event->event_manual_color = $colorsEventsGoogle[$colorId];
                        $event->event_manual_color = $schedulesData[$eventGoogle['calendarID']]['backgroundColor'];

                        // Se nao obter a cor nem na agenda, seta uma cor padrao por garantia
                        if(empty($event->event_manual_color)){
                            $event->event_manual_color = '#7986CB';
                        }
                    }
                }
            }

            // echo 'Evento: <pre>';
            // var_dump($event);
            // exit;

            // Salva pelo id do modulo e id do calendar
            if ((!empty($idModule)) && (!empty($idCalendar))){
                self::saveEvent($event, $idCalendar, $idModule);
            }
            // Salva apenas pelo id do modulo
            elseif(!empty($idModule)){
                self::saveEvent($event, '', $idModule);
            }
            // Salva apenas pelo id do calendar
            elseif(!empty($idCalendar)){
                self::saveEvent($event, $idCalendar);
            }
        }

        return $events;
    }

    /**
     * Atualiza data da ultima importacao automatica de eventos
	 *
     * @param   Int         $idModule 		    Id do modulo
     * @param   String      $dateString 	    Data a atualizar
     * @param   String      $aliasModalApi      Alias da modal de api que esta sendo atualizada (ex:'google_calendar')
	 *
	 * @return  Mixed       True ou false
	 */
    private static function apiUpdatesAutomaticImportDate($idModule, $dateString, $aliasModalApi){
        jimport('noboss.util.modules');

        // Obtem parametros do modulo
        $dataParams = NoBossUtilModules::getDataModule($idModule, true);

        // Extrai somente os parametros da modal de api
        $paramsModalApi = json_decode($dataParams->{$aliasModalApi});

        // Atualiza data da importacao
        $paramsModalApi->importer_automatic_date = $dateString;

        // Encoda novamente os parametros
        $paramsModalApi = json_encode($paramsModalApi);
        $dataParams->{$aliasModalApi} = $paramsModalApi;

        // Atualiza no banco
        if(NoBossUtilModules::setDataModule($idModule, $dataParams, true)){
            return true;
        }
        return false;
    }

    /**
	 * Remove eventos do google da base de dados para um periodo informado
     * 
     * @param   String      $startDate 		Data inicial no formato aaaa-dd-mm
     * @param   String      $finalDate 		Data final no formato aaaa-dd-mm
     * @param   String      $idCalendar 	Id do calendario (opcional, se informado idModule)
     * @param   Int         $idModule 		Id do modulo (opcional, se informado idCalendar)
	 *
	 * @return  Mixed       True ou false
	 */
    private static function apiRemovePeriodEvents($api, $startDate, $finalDate, $idCalendar = 0, $idModule = 0){
        $db = JFactory::getDBO();
        $query = $db->getQuery(true);

        // Remove do banco todos os eventos do google que estejam vinculados ao modulo e estejam dentro do periodo de importacao
        $query->delete('#__noboss_calendar');
        $query->where("initial_date >= '{$startDate}'");
        $query->where("initial_date <= '{$finalDate}'");
        $query->where('input_type =' . $db->quote($api));

        // Remove pelo id do calendar
        if(!empty($idCalendar)){
            $query->where('id_calendar =' . $db->quote($idCalendar));
        }
        else{
            return false;
        }
        
        //echo str_replace('#__', 'ext_', $query); exit;

        try {
            $db->setQuery($query);
            $db->execute();
        } catch (Exception $e) {
            exit('Error removing events from the database before importing. <br /> Details: '.$e->getMessage());
        }
    }
}
